<?php
require_once 'db_connection_pdo.php';

// Fetch filter options
$documentOptions = ['Invoice', 'Job Card', 'Rent', 'IN_EX', 'CUS_RTN'];
$paymentTypes = ['Cash', 'Credit Card', 'Online Transfer']; // Adjust according to your data

// Initialize filter variables
$startDate = isset($_GET['start_date']) ? $_GET['start_date'] : '';
$endDate = isset($_GET['end_date']) ? $_GET['end_date'] : '';
$selectedDocument = isset($_GET['document']) ? $_GET['document'] : '';
$selectedPaymentType = isset($_GET['payment_type']) ? $_GET['payment_type'] : '';

// Convert input dates to 'Y-m-d' format (if they are provided)
$startDateFormatted = $startDate ? DateTime::createFromFormat('Y-m-d', $startDate)->format('Y-m-d') : '';
$endDateFormatted = $endDate ? DateTime::createFromFormat('Y-m-d', $endDate)->format('Y-m-d') : '';

// Build SQL query with filters
$sql = "
SELECT 
    Document, Document_No, Disp,  DATE_FORMAT(Date, '%d-%m-%Y') AS Date, P_Type, Amount_In, Amount_Out
FROM (
    SELECT 
        'Invoice' AS Document, ipd.invoice_no AS Document_No,'' as Disp, ipd.payment_date AS Date, ipd.payment_type AS P_Type, ipd.payment AS Amount_In, NULL AS Amount_Out
    FROM invoice_payment_details ipd
    
    UNION ALL
    
    SELECT 
        'Job Card' AS Document, jp.job_card_id AS Document_No,'' as Disp, jp.date AS Date, jp.payment_by AS P_Type, jp.payment_amount AS Amount_In, NULL AS Amount_Out
    FROM job_payments jp
    
    UNION ALL
    
    SELECT 
        'Rent' AS Document, rp.rent_no AS Document_No,'' as Disp, rp.date AS Date, rp.payment_type AS P_Type, rp.payment AS Amount_In, NULL AS Amount_Out
    FROM rent_payment rp
    
    UNION ALL
    
    SELECT 
        'IN_EX' AS Document, ie.id AS Document_No, ie.description as Disp, ie.date AS Date, 'Cash' AS P_Type, ie.amount AS Amount_In, NULL AS Amount_Out
    FROM income_expenses ie
    WHERE ie.status = 'In'
    
    UNION ALL
    
    SELECT 
        'IN_EX' AS Document, ie.id AS Document_No, ie.description as Disp, ie.date AS Date, 'Cash' AS P_Type, ie.amount AS Amount_In, NULL AS Amount_Out
    FROM income_expenses ie
    WHERE ie.status = 'Opening Balance'
    
    UNION ALL

    SELECT 
        'IN_EX' AS Document, ie.id AS Document_No, ie.description as Disp, ie.date AS Date, 'Cash' AS P_Type, NULL AS Amount_In, ie.amount AS Amount_Out
    FROM income_expenses ie
    WHERE ie.status = 'Out'
    
    UNION ALL
    
    SELECT 
        'CUS_RTN' AS Document, crm.cus_rtn_id AS Document_No, '' as Disp, crm.date AS Date, 'Cash' AS P_Type, NULL AS Amount_In, SUM(cd.subtotal) AS Amount_Out
    FROM customer_return_master crm
    INNER JOIN customer_details cd ON crm.cus_rtn_id = cd.cus_rtn_id
    GROUP BY crm.cus_rtn_id, crm.date
) AS combined_report
WHERE 1=1
";

// Apply filters
$filters = [];
if ($startDateFormatted) {
    $sql .= " AND Date >= :start_date";
    $filters[':start_date'] = $startDateFormatted;
}
if ($endDateFormatted) {
    $sql .= " AND Date <= :end_date";
    $filters[':end_date'] = $endDateFormatted;
}
if ($selectedDocument) {
    $sql .= " AND Document = :document";
    $filters[':document'] = $selectedDocument;
}
if ($selectedPaymentType) {
    $sql .= " AND (P_Type = :payment_type OR P_Type IS NULL)";
    $filters[':payment_type'] = $selectedPaymentType;
}

// Execute query
$stmt = $pdo->prepare($sql);
$stmt->execute($filters);
$results = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Initialize total counters
$totalAmountIn = 0;
$totalAmountOut = 0;

// Calculate totals
foreach ($results as $row) {
    $totalAmountIn += $row['Amount_In'] ?? 0;
    $totalAmountOut += $row['Amount_Out'] ?? 0;
}

// Calculate balance
$balance = $totalAmountIn - $totalAmountOut;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Income and Expenses Report</title>
    <!-- Bootstrap CSS -->
    <link href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
<?php require_once 'menu.html';?>
    <div class="container mt-5">
        <h1 class="mb-4">Income and Expenses Report</h1>
        
        <form method="GET" class="mb-4">
            <div class="form-row">
                <div class="form-group col-md-3">
                    <label for="start_date">Start Date:</label>
                    <input type="date" class="form-control" id="start_date" name="start_date" value="<?php echo htmlspecialchars($startDate); ?>">
                </div>
                <div class="form-group col-md-3">
                    <label for="end_date">End Date:</label>
                    <input type="date" class="form-control" id="end_date" name="end_date" value="<?php echo htmlspecialchars($endDate); ?>">
                </div>
                <div class="form-group col-md-3">
                    <label for="document">Document:</label>
                    <select id="document" name="document" class="form-control">
                        <option value="">All</option>
                        <?php foreach ($documentOptions as $option): ?>
                            <option value="<?php echo $option; ?>" <?php if ($option == $selectedDocument) echo 'selected'; ?>>
                                <?php echo $option; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group col-md-3">
                    <label for="payment_type">Payment Type:</label>
                    <select id="payment_type" name="payment_type" class="form-control">
                        <option value="">All</option>
                        <?php foreach ($paymentTypes as $type): ?>
                            <option value="<?php echo $type; ?>" <?php if ($type == $selectedPaymentType) echo 'selected'; ?>>
                                <?php echo $type; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            <button type="submit" class="btn btn-primary">Filter</button>
        </form>

        <table class="table table-striped">
            <thead class="thead-dark">
                <tr>
                    <th>Document</th>
                    <th>Document No</th>
                    <th>DIS</th>
                    <th>Date</th>
                    <th>Payment Type</th>
                    <th>Amount In</th>
                    <th>Amount Out</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($results) > 0): ?>
                    <?php foreach ($results as $row): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['Document'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($row['Document_No'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($row['Disp'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($row['Date'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($row['P_Type'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars(number_format($row['Amount_In'] ?? 0, 2)); ?></td>
                            <td><?php echo htmlspecialchars(number_format($row['Amount_Out'] ?? 0, 2)); ?></td><td>
              
                           


                            
                            

                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="6" class="text-center">No results found.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
            <tfoot>
                <tr>
                    <th colspan="4" class="text-right">Total:</th>
                    <th><?php echo htmlspecialchars(number_format($totalAmountIn, 2)); ?></th>
                    <th><?php echo htmlspecialchars(number_format($totalAmountOut, 2)); ?></th>
                </tr>
                <tr>
                    <th colspan="4" class="text-right">Balance:</th>
                    <th colspan="2"><?php echo htmlspecialchars(number_format($balance, 2)); ?></th>
                </tr>
            </tfoot>
        </table>
    </div>

    <!-- Bootstrap JS and dependencies (Optional) -->
    <script src="https://code.jquery.com/jquery-3.5.1.slim.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.9.2/dist/umd/popper.min.js"></script>
    <script src="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
</body>
</html>
